import OpenAI from "openai";

const SYSTEM =
  "You are Cocobot ($COCO) — an AI agent at the intersection of chat and blockchain. You are helpful, precise, and slightly technical. You never ask for or expose private keys. In this web chat you can only talk; you don't have access to wallet or chain. If the user asks for balance or send, tell them they can run `npx cocobot agent` on their PC to use those skills. Be concise and friendly.";

export default {
  async fetch(request: Request): Promise<Response> {
    if (request.method !== "POST") {
      return new Response(JSON.stringify({ error: "Method not allowed" }), {
        status: 405,
        headers: { "Content-Type": "application/json" },
      });
    }
    const key = process.env.OPENAI_API_KEY;
    if (!key) {
      return new Response(
        JSON.stringify({ error: "OPENAI_API_KEY not set on the server." }),
        { status: 500, headers: { "Content-Type": "application/json" } }
      );
    }
    let body: { message?: string; history?: { role: string; content: string }[] };
    try {
      body = await request.json();
    } catch {
      return new Response(JSON.stringify({ error: "Invalid JSON" }), {
        status: 400,
        headers: { "Content-Type": "application/json" },
      });
    }
    const message = body?.message;
    if (!message || typeof message !== "string") {
      return new Response(JSON.stringify({ error: "message required" }), {
        status: 400,
        headers: { "Content-Type": "application/json" },
      });
    }
    const history = Array.isArray(body.history)
      ? body.history
          .filter(
            (m: unknown) =>
              m &&
              typeof m === "object" &&
              "role" in m &&
              "content" in m &&
              (m as { role: string }).role !== "system"
          )
          .slice(-20)
          .map((m: { role: string; content: string }) => ({
            role: m.role === "assistant" ? "assistant" : "user",
            content: String(m.content),
          }))
      : [];

    const openai = new OpenAI({ apiKey: key });
    const messages: OpenAI.Chat.ChatCompletionMessageParam[] = [
      { role: "system", content: SYSTEM },
      ...(history as OpenAI.Chat.ChatCompletionMessageParam[]),
      { role: "user", content: message.trim() },
    ];
    try {
      const completion = await openai.chat.completions.create({
        model: process.env.OPENAI_CHAT_MODEL || "gpt-4o-mini",
        messages,
        max_tokens: 1024,
      });
      const reply =
        completion.choices[0]?.message?.content?.trim() ?? "No response.";
      return new Response(JSON.stringify({ reply }), {
        status: 200,
        headers: {
          "Content-Type": "application/json",
          "Access-Control-Allow-Origin": "*",
        },
      });
    } catch (e) {
      const msg = e instanceof Error ? e.message : "OpenAI request failed";
      return new Response(JSON.stringify({ error: msg }), {
        status: 500,
        headers: { "Content-Type": "application/json" },
      });
    }
  },
};
