import * as fs from "fs";
import * as path from "path";
import { createInterface } from "readline";
import { getWalletBalance, getTokenBalance, sendNative } from "./skills/wallet.js";
import { config, hasWallet } from "./config.js";
import type { Address } from "viem";

const WORKSPACE = config.workspacePath;

function loadBootstrap(): string {
  const files = ["AGENTS.md", "SOUL.md", "IDENTITY.md", "TOOLS.md"];
  let out = "";
  for (const f of files) {
    const p = path.join(WORKSPACE, f);
    if (fs.existsSync(p)) {
      out += `\n--- ${f} ---\n` + fs.readFileSync(p, "utf-8");
    }
  }
  return out;
}

async function runTool(name: string, args: string[]): Promise<string> {
  try {
    if (name === "wallet_balance") {
      const result = await getWalletBalance();
      let msg = `Address: ${result.address}\nNative: ${result.native} ETH (chainId: ${result.chainId})`;
      if (config.cocoTokenAddress) {
        const tok = await getTokenBalance(config.cocoTokenAddress);
        msg += `\n$COCO: ${tok.balance} ${tok.symbol}`;
      }
      return msg;
    }
    if (name === "token_balance" && args.length >= 1) {
      const tokenAddress = args[0] as Address;
      const account = args[1] as Address | undefined;
      const tok = await getTokenBalance(tokenAddress, account);
      return `${tok.symbol}: ${tok.balance}`;
    }
    if (name === "wallet_send" && args.length >= 2) {
      const [to, amount] = args;
      const result = await sendNative(to as Address, amount);
      return `Sent. Tx: ${result.hash}`;
    }
    return `Unknown tool or missing args: ${name} ${args.join(" ")}`;
  } catch (e) {
    return `Error: ${e instanceof Error ? e.message : String(e)}`;
  }
}

function parseCommand(line: string): { tool: string; args: string[] } | null {
  const t = line.trim();
  if (t.startsWith("/balance") || t === "balance") {
    return { tool: "wallet_balance", args: [] };
  }
  const sendMatch = t.match(/^\/(?:send)\s+(0x[a-fA-F0-9]{40})\s+([\d.]+)\s*$/) ?? t.match(/^send\s+(0x[a-fA-F0-9]{40})\s+([\d.]+)\s*$/);
  if (sendMatch) return { tool: "wallet_send", args: [sendMatch[1], sendMatch[2]] };
  if (t.startsWith("/token ") && t.split(/\s+/).length >= 2) {
    const parts = t.split(/\s+/);
    return { tool: "token_balance", args: parts.slice(1) };
  }
  return null;
}

export async function runAgentLoop(): Promise<void> {
  const bootstrap = loadBootstrap();
  console.log("🥥 Cocobot ($COCO) — Agent + Blockchain\n");
  if (bootstrap) console.log("[Loaded workspace context]\n");
  if (!hasWallet()) console.log("⚠ No COCO_WALLET_PRIVATE_KEY set. Wallet tools disabled.\n");
  console.log("Commands: /balance | /send <address> <amount> | /token <tokenAddress> [account] | /quit\n");

  const rl = createInterface({ input: process.stdin, output: process.stdout });
  const prompt = () => process.stdout.write("$COCO> ");

  prompt();
  rl.on("line", async (line) => {
    const cmd = parseCommand(line);
    if (line.trim() === "/quit" || line.trim() === "exit") {
      rl.close();
      process.exit(0);
    }
    if (cmd) {
      const result = await runTool(cmd.tool, cmd.args);
      console.log(result);
    } else if (line.trim()) {
      console.log("(No LLM connected. Use /balance, /send <addr> <amount>, /token <addr> [account], or /quit)");
    }
    prompt();
  });
}
