import OpenAI from "openai";

function getClient(): OpenAI {
  const key = process.env.OPENAI_API_KEY;
  if (!key) throw new Error("OPENAI_API_KEY not set. Add it to .env to enable chat.");
  return new OpenAI({ apiKey: key });
}

export type ChatMessage = { role: "user" | "assistant"; content: string };

export async function chat(
  userMessage: string,
  systemPrompt: string,
  history: ChatMessage[] = []
): Promise<string> {
  const openai = getClient();

  const messages: OpenAI.Chat.ChatCompletionMessageParam[] = [
    { role: "system", content: systemPrompt },
    ...history.slice(-20).map((m) => ({ role: m.role, content: m.content })),
    { role: "user", content: userMessage },
  ];

  const completion = await openai.chat.completions.create({
    model: process.env.OPENAI_CHAT_MODEL || "gpt-4o-mini",
    messages,
    max_tokens: 1024,
  });

  return completion.choices[0]?.message?.content?.trim() ?? "No response.";
}
