#!/usr/bin/env node
import { runAgentLoop } from "./agent.js";
import { getWalletBalance, getTokenBalance, sendNative } from "./skills/wallet.js";
import { config, hasWallet } from "./config.js";

const args = process.argv.slice(2);
const cmd = args[0];

async function main() {
  if (cmd === "agent") {
    await runAgentLoop();
    return;
  }

  if (cmd === "balance") {
    if (!hasWallet()) {
      console.error("Set COCO_WALLET_PRIVATE_KEY in .env");
      process.exit(1);
    }
    const r = await getWalletBalance();
    console.log("Address:", r.address);
    console.log("Native:", r.native, "ETH (chainId:", r.chainId + ")");
    if (config.cocoTokenAddress) {
      const tok = await getTokenBalance(config.cocoTokenAddress);
      console.log("$COCO:", tok.balance, tok.symbol);
    }
    return;
  }

  if (cmd === "send" && args.length >= 3) {
    if (!hasWallet()) {
      console.error("Set COCO_WALLET_PRIVATE_KEY in .env");
      process.exit(1);
    }
    const [to, amount] = args.slice(1, 3);
    const result = await sendNative(to as `0x${string}`, amount);
    console.log("Tx:", result.hash);
    return;
  }

  if (cmd === "token" && args.length >= 2) {
    const tokenAddr = args[1] as `0x${string}`;
    const account = args[2] as `0x${string}` | undefined;
    const tok = await getTokenBalance(tokenAddr, account);
    console.log(tok.symbol + ":", tok.balance);
    return;
  }

  console.log(`
$COCO — Cocobot (Agent + Blockchain)

Usage:
  coco agent              Start agent loop (chat + /balance, /send, /token)
  coco balance            Show wallet balance (native + $COCO if configured)
  coco send <to> <amount> Send native token (e.g. 0.01 ETH)
  coco token <addr> [account]  ERC-20 balance for token

Env: COCO_RPC_URL, COCO_WALLET_PRIVATE_KEY, COCO_TOKEN_ADDRESS (optional)
Workspace: ./workspace (IDENTITY, SOUL, AGENTS, skills)
`);
}

main().catch((e) => {
  console.error(e);
  process.exit(1);
});
