import "dotenv/config";
import express, { type Request, type Response } from "express";
import path from "path";
import { fileURLToPath } from "url";
import fs from "fs";
import { chat } from "./chat.js";

const __dirname = path.dirname(fileURLToPath(import.meta.url));
const rootDir = path.join(__dirname, "..");
const publicDir = path.join(rootDir, "public");

const app = express();
app.use(express.json());

const COCO_SYSTEM = `You are Cocobot ($COCO) — an AI agent at the intersection of chat and blockchain. You are helpful, precise, and slightly technical. You never ask for or expose private keys. In this web chat you can only talk; you don't have access to wallet or chain. If the user asks for balance or send, tell them they can download Cocobot for PC to use those skills. Be concise and friendly.`;

app.use(express.static(rootDir));
app.use(express.static(publicDir));

app.get("/coco.png", (_req: Request, res: Response) => {
  const p = path.join(rootDir, "coco.png");
  if (fs.existsSync(p)) res.sendFile(p);
  else res.status(404).send("Not found");
});

app.post("/api/chat", async (req: Request, res: Response) => {
  const { message, history = [] } = req.body ?? {};
  if (!message || typeof message !== "string") {
    return res.status(400).json({ error: "message required" });
  }
  const safeHistory = Array.isArray(history)
    ? history
        .filter((m) => m && (m.role === "user" || m.role === "assistant") && typeof m.content === "string")
        .slice(-20)
    : [];
  try {
    const reply = await chat(message.trim(), COCO_SYSTEM, safeHistory);
    res.json({ reply });
  } catch (e) {
    console.error(e);
    res.status(500).json({
      error: e instanceof Error ? e.message : "Chat failed",
    });
  }
});

app.get("/download", (_req: Request, res: Response) => {
  const zipPath = path.join(publicDir, "cocobot.zip");
  if (!fs.existsSync(zipPath)) {
    return res.status(404).send("Run 'npm run pack' to generate the download package.");
  }
  res.setHeader("Content-Disposition", 'attachment; filename="cocobot.zip"');
  res.sendFile(zipPath);
});
app.get("/cocobot.zip", (_req: Request, res: Response) => {
  const zipPath = path.join(publicDir, "cocobot.zip");
  if (!fs.existsSync(zipPath)) {
    return res.status(404).send("Run 'npm run pack' first.");
  }
  res.setHeader("Content-Disposition", 'attachment; filename="cocobot.zip"');
  res.sendFile(zipPath);
});

const port = Number(process.env.PORT) || 3000;
app.listen(port, () => {
  console.log(`🥥 Cocobot landing: http://localhost:${port}`);
  if (!process.env.OPENAI_API_KEY) console.warn("⚠ OPENAI_API_KEY not set — chat will fail.");
});
