import { createPublicClient, createWalletClient, http, formatEther, parseEther, getAddress } from "viem";
import { privateKeyToAccount } from "viem/accounts";
import { config, hasWallet } from "../config.js";
import { base } from "../chains.js";

const transport = http(config.rpcUrl);

export async function getWalletBalance(accountAddress?: `0x${string}`): Promise<{
  address: string;
  native: string;
  nativeWei: bigint;
  chainId: number;
}> {
  const client = createPublicClient({ transport, chain: base });
  const chainId = (await client.getChainId()) ?? 8453;
  const address = accountAddress ?? (hasWallet() ? getWalletAccount().address : undefined);
  if (!address) throw new Error("No wallet configured. Set COCO_WALLET_PRIVATE_KEY or pass accountAddress.");
  const nativeWei = await client.getBalance({ address });
  return {
    address: getAddress(address),
    native: formatEther(nativeWei),
    nativeWei,
    chainId,
  };
}

export async function getTokenBalance(
  tokenAddress: `0x${string}`,
  accountAddress?: `0x${string}`
): Promise<{ symbol: string; decimals: number; balance: string; raw: bigint }> {
  const client = createPublicClient({ transport, chain: base });
  const address = accountAddress ?? (hasWallet() ? getWalletAccount().address : undefined);
  if (!address) throw new Error("No wallet configured.");
  const [decimals, symbol, balance] = await Promise.all([
    client.readContract({
      address: tokenAddress,
      abi: [{ name: "decimals", type: "function", stateMutability: "view", inputs: [], outputs: [{ type: "uint8" }] }],
      functionName: "decimals",
    }),
    client.readContract({
      address: tokenAddress,
      abi: [{ name: "symbol", type: "function", stateMutability: "view", inputs: [], outputs: [{ type: "string" }] }],
      functionName: "symbol",
    }),
    client.readContract({
      address: tokenAddress,
      abi: [{ name: "balanceOf", type: "function", stateMutability: "view", inputs: [{ name: "account", type: "address" }], outputs: [{ type: "uint256" }] }],
      functionName: "balanceOf",
      args: [address],
    }),
  ]);
  const divisor = 10n ** BigInt(decimals);
  const balanceStr = (Number(balance) / Number(divisor)).toFixed(decimals);
  return { symbol: symbol as string, decimals, balance: balanceStr, raw: balance };
}

export async function sendNative(to: `0x${string}`, amountEth: string): Promise<{ hash: `0x${string}`; success: boolean }> {
  if (!hasWallet()) throw new Error("No wallet configured. Set COCO_WALLET_PRIVATE_KEY.");
  const account = getWalletAccount();
  const walletClient = createWalletClient({
    account,
    transport,
    chain: base,
  });
  const hash = await walletClient.sendTransaction({
    to: getAddress(to),
    value: parseEther(amountEth),
    chain: base,
  });
  if (!hash) throw new Error("Send failed.");
  return { hash, success: true };
}

function getWalletAccount() {
  if (!config.privateKey) throw new Error("COCO_WALLET_PRIVATE_KEY not set.");
  return privateKeyToAccount(config.privateKey);
}

export function walletAddress(): string | undefined {
  if (!hasWallet()) return undefined;
  return getWalletAccount().address;
}
